# Arborform

## Introduction

Pawpaw supports the rapid creation of complex text parsers using the arborform module.  Arborform consists of two class hierarchies:

``Itorator``[^itorator_name]: Generator-based pipelined ``Ito`` creation 
``Postorator``: Pipelined ``Ito`` sequence consolidation

## The Pipeline

Pawpaw's arborform supports rapid parser creation using a divide and conquer strategy composed of discrete, pipelined steps.  Steps are peformed by ``Itorator`` instances, which operate on an ``Ito`` sequence.  An ``Itorator``, at the most basic level, takes an ``Ito`` and transforms into a new ``Ito`` *sequence* via its ``.iter``: method

```mermaid
stateDiagram-v2
  direction TB
  iter: itorator.iter()
  state iter {
    state "ito: Ito" as ito
    state "itos: typing.Sequence[Ito]" as itos
    ito --> itos
  }
```

The  ``.traverse`` method serves as a pipeline starting point, taking a single ``Ito`` as input:

```python
>>> from pawpaw import Ito, arborform
>>> s = 'The quick brown fox'
>>> i = Ito(s)
>>> split_words = arborform.Itorator.wrap(lambda ito: ito.str_split())  # perform str.split
>>> for result in split_words.traverse(i):
>>>   print(result)
The
quick
brown
fox
```

Itorator's are chained together via the ``itor_next`` and ``itor_children`` properties.  If "itor_next" exists, the results of the first ``Itorator`` are used as inputs for the next, and so on, until ``itor_next`` is None, at which point the sequence is returned:

```python
>>> split_chars = arborform.Itorator.wrap(lambda ito: [*ito])  # convert to 1-char long Itos
>>> split_words.itor_next = split_chars
>>> for result in split_words.traverse(i):
>>>   print(result)
T
h
e
q
u
i
c
k
b
r
o
w
n
f
o
x
```

Itorator's can alos be chained together via the ``itor_children`` property.  If "itor_children" exists, the results of the first ``Itorator`` are used fed as inputs to it, and results are then *appended* to the children:

```python
>>> split_words.itor_next = None  # clear pipeline
>>> split_words.itor_children = split_chars
>>> tree_vis = pawpaw.visualization.pepo.Tree()
>>> for result in split_words.traverse(i):
>>>   print(tree_vis.dumps(result)
(0, 3) 'None' : 'The'
├──(0, 1) 'None' : 'T'
├──(1, 2) 'None' : 'h'
└──(2, 3) 'None' : 'e'

(4, 9) 'None' : 'quick'
├──(4, 5) 'None' : 'q'
├──(5, 6) 'None' : 'u'
├──(6, 7) 'None' : 'i'
├──(7, 8) 'None' : 'c'
└──(8, 9) 'None' : 'k'

(10, 15) 'None' : 'brown'
├──(10, 11) 'None' : 'b'
├──(11, 12) 'None' : 'r'
├──(12, 13) 'None' : 'o'
├──(13, 14) 'None' : 'w'
└──(14, 15) 'None' : 'n'

(16, 19) 'None' : 'fox'
├──(16, 17) 'None' : 'f'
├──(17, 18) 'None' : 'o'
└──(18, 29) 'None' : 'x'
```

The arboform pipeline for a chain of itorators is as follows:

```mermaid
stateDiagram-v2
  direction LR
  
  [*] --> itorator
  state itorator {
    direction TB
    [*] --> itosP
    itosP : itos
    itosP --> [*]
  }
  itorator --> for_each  
  state for_each {
    direction LR
    [*] --> itor_children
    state itor_children {
      direction TB
      [*] --> itosC
      itosC: itos
      as_children: add to .children
      itosC --> as_children
}
    itor_children --> itor_next
    state itor_next {
      direction TB
      [*] --> itosN
      itosN: itos
      itosN --> [*]
    }
    itor_next --> [*]
  }
  for_each --> [*]
``` 
   
## Class Hierarchy

```mermaid
classDiagram
  class Itorator{
      +itor_next Itorator | Types.F_ITO_2_ITOR | None
      +itor_children Itorator | Types.F_ITO_2_ITOR | None
      +postorator Postorator | Types.F_ITOS_2_BITOS | None
      +_iter(ito) C_IT_ITOS
      +traverse(ito) C_IT_ITOS
  }
  
  Itorator <|-- Reflect
  Itorator <|-- Desc
  Itorator <|-- Group
  Itorator <|-- Split
  Itorator <|-- Extract

  class Postorator{
      +traverse(itos) C_IT_BITOS
  }

  Postorator <|-- stacked_reduce
  Postorator <|-- windowed_join
```

[^itorator_name]: The name "Itorator" comes from a portmanteau of "Ito" and "Iterator"
